#!/usr/bin/env python3
# computex_miner/plugins/pow_miner.py
"""
Frontier-only PoW worker (Bitcoin-style double-SHA256).
- Accepts payload either flat or wrapped under {"task": {...}}.
- Supports 76-byte headers (append 4-byte nonce) and 80-byte headers (overwrite).
- Optional layout fields:
    nonce_offset (default: 76), nonce_size (default: 4), nonce_little_endian (default: True)
- Periodic hashrate logs; clean error messages.
- Unified progress reporting via chunk["_report_progress"](current, end, status).
"""

import sys
import time
import os  # NEW

# Make this discoverable by the universal miner:
HANDLER_ALIASES = ["pow"]

# ---- Debug toggle ----
POW_DEBUG = os.getenv("COMPUTEX_DEBUG", "").lower() in ("1", "true", "yes")


def pow_debug(msg: str) -> None:
    if POW_DEBUG:
        print(msg, flush=True)


pow_debug(f"[pow_miner] sys.executable: {sys.executable}")
try:
    import hashlib
    pow_debug("[pow_miner] hashlib OK")
except Exception as e:
    # Import error is real and should be visible
    print("[pow_miner] import error (hashlib):", e)


# ---------------------------------------------------------------------
# Discovery helper (lets the launcher auto-pick this plugin)
# ---------------------------------------------------------------------
def can_handle(task: dict) -> bool:
    if not isinstance(task, dict):
        return False
    # Accept if explicitly hinted
    for k in ("handler", "job_kind", "task_kind", "kind", "type", "job_type"):
        v = task.get(k) or task.get("task", {}).get(k)
        if isinstance(v, str) and v.strip().lower() == "pow":
            return True
    # Or if it smells like a PoW task
    body = task.get("payload") if isinstance(task.get("payload"), dict) else task
    return bool(isinstance(body, dict) and body.get("block_header") and body.get("target"))

# ---------------------------------------------------------------------
# Helpers
# ---------------------------------------------------------------------
def double_sha256(data_bytes: bytes) -> bytes:
    """Double SHA256, as in Bitcoin mining."""
    return hashlib.sha256(hashlib.sha256(data_bytes).digest()).digest()

def _parse_int(v, default):
    try:
        return int(v)
    except Exception:
        return default

def _parse_bool(v, default=False):
    if isinstance(v, bool):
        return v
    if isinstance(v, str):
        return v.strip().lower() in ("1", "true", "yes", "on")
    return default

def _strip0x(h: str) -> str:
    return h[2:] if isinstance(h, str) and h.lower().startswith("0x") else (h or "")

def _resolve_task(chunk):
    """Allow frontier payload styles:
       - flat: {"block_header":..., "target":..., ...}
       - wrapped: {"task": {"payload": {...}}} or {"task": {...}} or {"payload": {...}}
    """
    if not isinstance(chunk, dict):
        return {}
    if "task" in chunk and isinstance(chunk["task"], dict):
        t = chunk["task"]
        if "payload" in t and isinstance(t["payload"], dict):
            # Merge common top-level convenience fields if present
            return {
                **t.get("payload", {}),
                **{k: v for k, v in t.items() if k in (
                    "block_header", "target", "nonce_start", "nonce_end",
                    "nonce_offset", "nonce_size", "nonce_little_endian"
                )}
            }
        return t
    if "payload" in chunk and isinstance(chunk["payload"], dict):
        return chunk["payload"]
    return chunk

# ---------------------------------------------------------------------
# Core mining loop
# ---------------------------------------------------------------------
def _mine(block_header_hex: str,
          target_hex: str,
          nonce_start: int,
          nonce_end: int,
          *,
          nonce_offset: int = 76,
          nonce_size:   int = 4,
          nonce_little_endian: bool = True,
          report_progress_cb=None):
    """
    If header is 76 bytes (152 hex), we append a 4-byte nonce.
    If header is 80 bytes (160 hex), we overwrite nonce at nonce_offset..offset+size.
    report_progress_cb: callable(done_nonces:int) -> None
    """
    if not isinstance(block_header_hex, str) or not isinstance(target_hex, str):
        print("[pow_miner] ERROR: header/target must be hex strings.", file=sys.stderr)
        return None, None

    bh_hex = _strip0x(block_header_hex).lower()
    tgt_hex = _strip0x(target_hex).lower()

    # Validate hex
    try:
        block_header_bytes = bytes.fromhex(bh_hex)
    except Exception:
        print("[pow_miner] ERROR: Invalid block_header hex string.", file=sys.stderr)
        return None, None
    try:
        target_int = int(tgt_hex, 16)
    except Exception:
        print("[pow_miner] ERROR: Invalid target hex string.", file=sys.stderr)
        return None, None

    header_len = len(block_header_bytes)
    if header_len not in (76, 80):
        print(f"[pow_miner] ERROR: block_header must be 76 or 80 bytes (got {header_len}).", file=sys.stderr)
        return None, None

    if nonce_size != 4:
        print("[pow_miner] WARN: nonce_size != 4 is uncommon; proceeding.", file=sys.stderr)

    # Prepare constant parts
    write_nonce = (header_len == 80)
    if write_nonce:
        # Overwrite into an 80-byte header
        if nonce_offset < 0 or nonce_offset + nonce_size > 80:
            print("[pow_miner] ERROR: nonce_offset/size out of range for 80-byte header.", file=sys.stderr)
            return None, None
        base_prefix = block_header_bytes[:nonce_offset]
        base_suffix = block_header_bytes[nonce_offset + nonce_size:]
    else:
        # 76-byte header: append nonce bytes each iteration
        base_prefix = block_header_bytes
        base_suffix = b""

    # Cadence
    last_log_nonce = nonce_start
    last_log_time = time.time()

    # Initial progress ping
    if report_progress_cb:
        try:
            report_progress_cb(0)
        except Exception:
            pass

    try:
        for nonce in range(nonce_start, nonce_end + 1):
            try:
                nb = nonce.to_bytes(nonce_size, byteorder=("little" if nonce_little_endian else "big"), signed=False)
            except OverflowError:
                print(f"[pow_miner] ERROR: Nonce {nonce} cannot fit in {nonce_size} bytes.", file=sys.stderr)
                return None, None

            header = (base_prefix + nb + base_suffix) if write_nonce else (base_prefix + nb)

            h = double_sha256(header)

            if int.from_bytes(h, "big") < target_int:
                if report_progress_cb:
                    try:
                        report_progress_cb(nonce - nonce_start + 1)
                    except Exception:
                        pass
                return nonce, h.hex()

            # Logs + progress pings
            if nonce == nonce_start or (nonce - last_log_nonce) >= 5_000_000 or (time.time() - last_log_time) >= 2.5:
                elapsed = max(1e-9, time.time() - last_log_time)
                hps = int((nonce - last_log_nonce + 1) / elapsed)
                print(f"[pow_miner] Mining... nonce={nonce} hash={h.hex()[:16]} ~{hps:,} H/s")
                last_log_nonce = nonce
                last_log_time = time.time()
                if report_progress_cb:
                    try:
                        report_progress_cb(nonce - nonce_start + 1)
                    except Exception:
                        pass

    except KeyboardInterrupt:
        print("[pow_miner] Interrupted by user.", file=sys.stderr)
        return None, None
    except Exception as e:
        print(f"[pow_miner] ERROR during mining: {e}", file=sys.stderr)
        return None, None

    return None, None

# ---------------------------------------------------------------------
# Entry point for universal_miner
# ---------------------------------------------------------------------
def run(chunk, args):
    """
    Frontier-only PoW worker.
    Expects payload to contain:
      - block_header (hex)  (76B without nonce, or 80B including nonce area)
      - target (hex)
      - nonce_start (int)
      - nonce_end   (int)
    Optional:
      - nonce_offset (default 76 if header is 80B)
      - nonce_size (default 4)
      - nonce_little_endian (default True)
    """
    job_id   = getattr(args, "job_id", "")
    lease_id = getattr(args, "lease_id", "")
    miner_id = getattr(args, "miner", None)

    payload = _resolve_task(chunk) or {}
    block_header = payload.get("block_header")
    target       = payload.get("target")
    nonce_start  = _parse_int(payload.get("nonce_start", 0), 0)
    nonce_end    = _parse_int(payload.get("nonce_end", 1_000_000), 1_000_000)

    # Optional layout
    nonce_offset         = _parse_int(payload.get("nonce_offset", 76), 76)
    nonce_size           = _parse_int(payload.get("nonce_size", 4), 4)
    nonce_little_endian  = _parse_bool(payload.get("nonce_little_endian", True), True)

    # Progress callback from universal_miner (if provided)
    progress_cb = chunk.get("_report_progress") if isinstance(chunk, dict) else None
    total = max(1, (nonce_end - nonce_start + 1))

    def report(current, end=total, status="running"):
        if not progress_cb:
            return
        try:
            progress_cb(int(current), int(end), status=status)
        except Exception:
            pass  # never break mining due to UI pings

    if not block_header or not target:
        print("[pow_miner] ERROR: block_header or target missing from payload.", file=sys.stderr)
        report(0, total, status="closed")  # close so UI re-enables
        return {
            "nonce": None,
            "hash": None,
            "block_header": block_header,
            "target": target,
            "nonce_start": nonce_start,
            "nonce_end": nonce_end,
            "status": "completed_no_solution",
            "message": "Missing block_header or target."
        }

    bh_preview = _strip0x(block_header)[:32]
    tgt_preview = _strip0x(target)[:16]
    print(f"[pow_miner] Frontier mining | job_id={job_id} lease_id={lease_id} "
          f"nonce_start={nonce_start} nonce_end={nonce_end} target={tgt_preview}... "
          f"header='{bh_preview}...' (offset={nonce_offset}, size={nonce_size}, LE={nonce_little_endian})")

    # Initial progress
    report(0, total, status="running")

    t0 = time.time()
    nonce, hash_hex = _mine(
        block_header, target, nonce_start, nonce_end,
        nonce_offset=nonce_offset,
        nonce_size=nonce_size,
        nonce_little_endian=nonce_little_endian,
        report_progress_cb=lambda done: report(done, total, status="running")
    )
    elapsed = time.time() - t0

    if nonce is not None:
        print(f"[pow_miner] ✅ solution found | nonce={nonce} hash={hash_hex} | {elapsed:.2f}s")
        report(total, total, status="closed")
        return {
            "nonce": nonce,
            "hash": hash_hex,
            "block_header": block_header,
            "target": target,
            "nonce_start": nonce_start,
            "nonce_end": nonce_end,
            "status": "completed_solution_found"
        }

    print(f"[pow_miner] ❌ no solution in range | {elapsed:.2f}s")
    report(total, total, status="closed")
    return {
        "nonce": None,
        "hash": None,
        "block_header": block_header,
        "target": target,
        "nonce_start": nonce_start,
        "nonce_end": nonce_end,
        "status": "completed_no_solution",
        "message": "No valid nonce found in this range."
    }

# Optional standalone test
if __name__ == "__main__":
    import argparse
    ap = argparse.ArgumentParser(description="Frontier PoW Miner (standalone test)")
    ap.add_argument("--block_header", required=True, help="76B (152 hex) without nonce, or 80B with nonce region")
    ap.add_argument("--target", required=True)
    ap.add_argument("--nonce_start", type=int, default=0)
    ap.add_argument("--nonce_end", type=int, default=1_000_000)
    ap.add_argument("--nonce_offset", type=int, default=76)
    ap.add_argument("--nonce_size", type=int, default=4)
    ap.add_argument("--nonce_little_endian", default="true")
    args = ap.parse_args()

    res = run({
        "block_header": args.block_header,
        "target": args.target,
        "nonce_start": args.nonce_start,
        "nonce_end": args.nonce_end,
        "nonce_offset": args.nonce_offset,
        "nonce_size": args.nonce_size,
        "nonce_little_endian": args.nonce_little_endian,
    }, argparse.Namespace(server="", job_id="", lease_id="", miner=None))
    print("[pow_miner] Final result:", res)
