import argparse
import importlib
import importlib.util
import json
import os
import sys
import time
import uuid
from typing import Dict, Optional, List, Callable

import requests

# ---------------------------------------------------------------------
# Constants / Paths
# ---------------------------------------------------------------------
ROOT_DIR = os.path.dirname(os.path.abspath(__file__))
PLUGIN_DIR = os.path.join(ROOT_DIR, "plugins")
os.makedirs(PLUGIN_DIR, exist_ok=True)

MINER_ID_FILE = os.path.join(ROOT_DIR, "miner_id.txt")

DEFAULT_TIMEOUT = 30
RETRY_ATTEMPTS = 3
RETRY_BACKOFF_SEC = 1.5  # exponential
BANNER = r"""
          ____                            _        __  __            
         / ___|___  _ __ ___  _ __  _   _| |_ ___  \ \/ /            
        | |   / _ \| '_ ` _ \| '_ \| | | | __/ _ \  \  /             
        | |__| (_) | | | | | | |_) | |_| | ||  __/  /  \             
         \____\___/|_| |_| |_| .__/ \__,_|\__\___| /_/\_\    
                             |_|
""".rstrip("\n")

# ---------------------------------------------------------------------
# Utilities
# ---------------------------------------------------------------------
def get_miner_id() -> str:
    """Stable, local miner id persisted to a file."""
    try:
        if os.path.exists(MINER_ID_FILE):
            with open(MINER_ID_FILE, "r", encoding="utf-8") as f:
                mid = f.read().strip()
                if mid:
                    return mid
        miner_id = uuid.uuid4().hex
        with open(MINER_ID_FILE, "w", encoding="utf-8") as f:
            f.write(miner_id)
        return miner_id
    except Exception as e:
        print("[universal_miner] WARN: could not persist miner id (%s); using ephemeral id." % e)
        return uuid.uuid4().hex


def load_payload(payload_file: str) -> dict:
    with open(payload_file, "r", encoding="utf-8") as f:
        return json.load(f)


def request_with_retries(
    method: str,
    url: str,
    *,
    attempts: int = RETRY_ATTEMPTS,
    backoff: float = RETRY_BACKOFF_SEC,
    timeout: int = DEFAULT_TIMEOUT,
    **kwargs
) -> requests.Response:
    """Make an HTTP request with simple exponential backoff."""
    last_err: Optional[Exception] = None
    for i in range(1, attempts + 1):
        try:
            resp = requests.request(method, url, timeout=timeout, **kwargs)
            return resp
        except Exception as e:
            last_err = e
            if i < attempts:
                sleep_for = backoff ** i
                print(
                    "[universal_miner] WARN: %s %s failed (attempt %d/%d): %s. "
                    "Retrying in %.1fs"
                    % (method, url, i, attempts, e, sleep_for)
                )
                time.sleep(sleep_for)
    assert last_err is not None
    raise last_err


# ---- Frontier progress hook -------------------------------------------------
def _progress_url(server: str, job_id: str, lease_id: str) -> str:
    # shadow id for Frontier leases
    shadow = "frontier::%s" % lease_id
    return "%s/jobs/%s/progress/%s" % (server.rstrip("/"), job_id, shadow)

def post_progress(server: str, job_id: str, lease_id: str,
                  current: int, end: int, status: str = "running"):
    """
    Unified progress ping.
    Prefer POST (most RESTful for updates); fall back to GET variants for legacy routes.
    """
    end = max(1, int(end))
    current = int(max(0, current))

    percent = 0
    try:
        percent = max(0, min(100, int((current / float(end)) * 100)))
    except Exception:
        percent = 0

    url = _progress_url(server, job_id, lease_id)
    payload = {"status": status, "current": current, "end": end, "percent": percent}

    # A) Preferred: POST /jobs/<id>/progress/frontier::<lease>
    try:
        request_with_retries("POST", url, json=payload, timeout=10)
        return
    except Exception:
        pass

    # B) Fallback: GET /jobs/<id>/progress/frontier::<lease>?...
    try:
        request_with_retries("GET", url, params=payload, timeout=10)
        return
    except Exception:
        pass

    # C) Fallback: GET /jobs/<id>/progress?chunk_index=frontier::<lease>&...
    try:
        alt = "%s/jobs/%s/progress" % (server.rstrip("/"), job_id)
        q = dict(payload)
        q["chunk_index"] = "frontier::%s" % lease_id
        request_with_retries("GET", alt, params=q, timeout=10)
        return
    except Exception as e:
        print("[universal_miner] WARN: all progress attempts failed: %s" % e)


# ---------------------------------------------------------------------
# Plugin discovery & import (dynamic)
# ---------------------------------------------------------------------
class PluginInfo:
    def __init__(self, name: str, module, aliases: List[str], can_handle: Optional[Callable[[dict], bool]]):
        self.name = name
        self.module = module
        self.aliases = aliases
        self.can_handle = can_handle


def _extract_aliases(mod) -> List[str]:
    for attr in ("HANDLER_ALIASES", "ALIASES", "HANDLER_ALIAS", "ALIASES_MAP"):
        val = getattr(mod, attr, None)
        if isinstance(val, (list, tuple)):
            return [str(a).lower() for a in val]
    base = getattr(mod, "__name__", "")
    if base.endswith("_miner"):
        return [base.rsplit(".", 1)[-1].replace("_miner", "")]
    return []


def discover_plugins() -> Dict[str, PluginInfo]:
    plugins: Dict[str, PluginInfo] = {}
    for fname in sorted(os.listdir(PLUGIN_DIR)):
        if not fname.endswith("_miner.py"):
            continue
        base = fname[:-3]
        path = os.path.join(PLUGIN_DIR, fname)
        spec = importlib.util.spec_from_file_location(base, path)
        if not spec or not spec.loader:
            print("[universal_miner] WARN: could not load plugin spec for %s" % fname)
            continue
        module = importlib.util.module_from_spec(spec)
        try:
            spec.loader.exec_module(module)  # type: ignore
        except Exception as e:
            print("[universal_miner] ERROR loading %s: %s" % (fname, e))
            continue
        aliases = _extract_aliases(module)
        can = getattr(module, "can_handle", None)
        info = PluginInfo(base, module, aliases, can if callable(can) else None)
        plugins[base] = info
        for a in aliases:
            plugins[a] = info
    

    return plugins


PLUGINS = discover_plugins()


def import_handler_module(name_or_alias: str):
    info = PLUGINS.get(name_or_alias)
    if info:
        return info.module
    # Try importing directly (PYTHONPATH)
    try:
        return importlib.import_module(name_or_alias)
    except Exception as e:
        print("[universal_miner] ERROR: Could not import handler '%s': %s" %
              (name_or_alias, e))
        print("[universal_miner] Known aliases: %s" %
              sorted(set(PLUGINS.keys())))
        sys.exit(2)


# ---------------------------------------------------------------------
# Handler inference (dynamic)
# ---------------------------------------------------------------------
def _payload_kind_hints(p: dict) -> List[str]:
    hints: List[str] = []
    for key in ("handler", "job_kind", "task_kind", "kind", "type", "job_type"):
        v = p.get(key) or p.get("task", {}).get(key) or p.get("job", {}).get(key)
        if isinstance(v, str) and v.strip():
            hints.append(v.strip().lower())
    return hints


def _probe_plugins_with_task(task: dict) -> Optional[str]:
    for unique_name in sorted(set(pi.name for pi in PLUGINS.values())):
        info = PLUGINS.get(unique_name)
        if not info:
            continue
        try:
            if info.can_handle and info.can_handle(task):
                return info.name
        except Exception as e:
            print("[universal_miner] WARN: %s.can_handle() errored: %s" %
                  (unique_name, e))
    return None


def _fetch_job_kind_from_server(server: str, job_id: str) -> str:
    try:
        url = "%s/jobs/%s" % (server.rstrip("/"), job_id)
        resp = request_with_retries("GET", url, timeout=10)
        resp.raise_for_status()
        job = resp.json()
        return (job.get("type") or job.get("kind") or "").strip().lower()
    except Exception as e:
        print("[universal_miner] WARN: failed to fetch job kind from server: %s" % e)
        return ""


def resolve_handler_module_name(cli_handler: Optional[str], payload: dict,
                                server: str, job_id: str) -> str:
    # 1) CLI flag
    if cli_handler:
        info = PLUGINS.get(cli_handler)
        if info:
            return info.name
        return cli_handler

    # 2) Payload hints
    for hint in _payload_kind_hints(payload or {}):
        info = PLUGINS.get(hint)
        if info:
            print("[universal_miner] Inferred handler from hint '%s' to '%s'" %
                  (hint, info.name))
            return info.name

    # 3) can_handle()
    modname = _probe_plugins_with_task(payload or {})
    if modname:
        print("[universal_miner] Selected by can_handle() to '%s'" % modname)
        return modname

    # 4) Ask server
    kind = _fetch_job_kind_from_server(server, job_id)
    if kind and PLUGINS.get(kind):
        print("[universal_miner] Fallback from job kind '%s' to '%s'" %
              (kind, PLUGINS[kind].name))
        return PLUGINS[kind].name

    # 5) Single plugin
    unique_modules = sorted(set(pi.name for pi in PLUGINS.values()))
    if len(unique_modules) == 1:
        print("[universal_miner] Single plugin available; using '%s'" %
              unique_modules[0])
        return unique_modules[0]

    print("[universal_miner] ERROR: Could not resolve a handler. "
          "Provide --handler explicitly.")
    print("[universal_miner] Known aliases: %s" %
          sorted(set(PLUGINS.keys())))
    sys.exit(2)


def infer_job_kind_for_log(payload: dict, server: str, job_id: str) -> str:
    # Prefer hints from the payload first
    hints = _payload_kind_hints(payload or {})
    if hints:
        return hints[0]
    return _fetch_job_kind_from_server(server, job_id) or ""


def print_job_header(server: str, miner_id: str,
                     job_id: str, lease_id: str, job_kind: str):
    print()
    print(BANNER)
    print("Miner session:")
    print("  server   : %s" % server)
    print("  miner    : %s" % miner_id)
    print("  job_id   : %s" % job_id)
    print("  lease_id : %s" % lease_id)
    if job_kind:
        print("  job_kind : %s" % job_kind)
    print("------------------------------------------------------------")
    print("Computation Result:")

# ---------------------------------------------------------------------
# Frontier submit
# ---------------------------------------------------------------------
def submit_result(args, miner_id: str, result: dict, job_kind: str = ""):
    payload = {
        "miner_id": miner_id,
        "lease_id": args.lease_id,
        "result": result or {"message": "No result returned from plugin."},
    }
    submit_url = "%s/jobs/%s/submit" % (args.server.rstrip("/"), args.job_id)
    try:
        resp = request_with_retries("POST", submit_url, json=payload, timeout=45)
        out = resp.json()
        # finalize progress (use plugin status if available)
        final_status = (result or {}).get("status") or "closed"
        post_progress(args.server, args.job_id, args.lease_id,
                      current=1, end=1, status=final_status)

        # ---- human-readable payout summary ----
        print("------------------------------------------------------------")
        print("Payout summary:")
        print("  ok          : %s" % out.get("ok"))
        print("  job_id      : %s" % out.get("job_id", args.job_id))
        print("  lease_id    : %s" % out.get("lease_id", args.lease_id))
        if job_kind:
            print("  job_kind    : %s" % job_kind)
        print("  chunk_index : %s" % out.get("chunk_index"))
        print("  verified    : %s" % out.get("verified"))
        print("  ctx_issued  : %s" % out.get("ctx_issued"))
        print("  base_ctx    : %s" % out.get("base_ctx"))
        print("  bonus_ctx   : %s" % out.get("bonus_ctx"))
        print("  new_balance : %s" % out.get("new_balance"))
        tx_hash = out.get("tx_hash") or out.get("transaction_hash")
        if tx_hash:
            print("  tx_hash     : %s" % tx_hash)
        explorer_url = out.get("explorer_url")
        if explorer_url:
            print("  explorer    : %s" % explorer_url)
        print("------------------------------------------------------------")

        if out.get("ok"):
            msg = out.get("message") or "OK"
            print("[universal_miner] Job finished successfully: %s" % msg)
        elif out.get("error") == "submit_error" and out.get("detail") == "minting_not_configured":
            print("[universal_miner] Piece solved, but payout is not configured on the server.")
        elif "error" in out:
            print("[universal_miner] Submit error: %s (HTTP %s)" %
                  (out["error"], resp.status_code))
        else:
            print("[universal_miner] Submit completed with unexpected response format.")
        return out

    except Exception as e:
        print("[universal_miner] Submit failed: %s" % e)
        # best effort: still tell UI we are done (error)
        post_progress(args.server, args.job_id, args.lease_id,
                      current=1, end=1, status="error")
        return None


# ---------------------------------------------------------------------
# Main (Frontier-only)
# ---------------------------------------------------------------------
def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--handler", required=False, help="Handler/job type alias or module name")
    ap.add_argument("--server", required=True)
    ap.add_argument("--miner", required=False, help="Miner ID for this worker (default: from miner_id.txt)")
    ap.add_argument("--job_id", required=True)
    ap.add_argument("--lease_id", required=True)
    ap.add_argument("--payload_file", required=True)

    args, unknown = ap.parse_known_args()
    if unknown:
        print("[universal_miner] WARN: ignoring unknown args: %s" % (unknown,))

    miner_id = args.miner or get_miner_id()

    try:
        chunk = load_payload(args.payload_file)
    except Exception as e:
        print("[universal_miner] ERROR: could not read payload file: %s" % e)
        sys.exit(2)

    if isinstance(chunk, dict) and chunk.get("error"):
        print("[universal_miner] Payload indicates error from server: %s" %
              chunk.get("error"))
        sys.exit(0)

    pl_lease = (chunk.get("lease_id") or chunk.get("task", {}).get("lease_id")) if isinstance(chunk, dict) else None
    if pl_lease and str(pl_lease) != str(args.lease_id):
        print("[universal_miner] WARN: payload lease_id (%s) != CLI lease_id (%s)" %
              (pl_lease, args.lease_id))

    # Infer job kind for logging
    payload_for_kind = chunk if isinstance(chunk, dict) else {}
    job_kind = infer_job_kind_for_log(payload_for_kind, args.server, args.job_id)

    # Nice header with all the key IDs
    print_job_header(args.server, miner_id, args.job_id, args.lease_id, job_kind)

    # Provide a progress callback to the plugin, including log of percent
    last_percent = {"value": -1}

    def report_progress(current: int, end: int, status: str = "running"):
        try:
            end_clamped = max(1, int(end))
            current_clamped = max(0, int(current))
            pct = 0
            try:
                pct = max(0, min(100, int((current_clamped / float(end_clamped)) * 100)))
            except Exception:
                pct = 0
            if pct != last_percent["value"]:
                print("[progress] job=%s lease=%s status=%s percent=%d (%d/%d)" %
                      (args.job_id, args.lease_id, status, pct, current_clamped, end_clamped))
                last_percent["value"] = pct
            post_progress(args.server, args.job_id, args.lease_id,
                          current_clamped, end_clamped, status)
        except Exception as _e:
            print("[universal_miner] WARN: progress post failed: %s" % _e)

    if isinstance(chunk, dict):
        chunk["_report_progress"] = report_progress

    # Resolve handler automatically if not provided
    handler_module_name = resolve_handler_module_name(args.handler, chunk or {}, args.server, args.job_id)
    handler = import_handler_module(handler_module_name)

    if not hasattr(handler, "run"):
        print("[universal_miner] Handler '%s' missing required 'run(chunk, args)' function." %
              handler_module_name)
        report_progress(0, 1, status="error")
        sys.exit(3)

    # PRIME progress so the UI always shows a bar immediately
    report_progress(0, 1, status="mining")

    try:
        print("[universal_miner] This install's Miner ID: %s" % miner_id)
        print("[universal_miner] Using handler: %s" % handler_module_name)
        result = handler.run(chunk, args)
        print("[universal_miner] Handler completed. Result: %s" % result)
        submit_result(args, miner_id, result, job_kind=job_kind)
        print("[universal_miner] DONE job=%s lease=%s" % (args.job_id, args.lease_id))
        sys.exit(0)
    except Exception as e:
        import traceback
        print("[universal_miner] ERROR: handler threw an exception: %s" % e)
        traceback.print_exc()
        report_progress(0, 1, status="error")
        sys.exit(3)


if __name__ == "__main__":
    main()
